/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// @format

/**
 * Project text search reducer
 * @module reducers/project-text-search
 */

import { createLocation } from "../utils/location";

export const statusType = {
  initial: "INITIAL",
  fetching: "FETCHING",
  cancelled: "CANCELLED",
  done: "DONE",
  error: "ERROR",
};

export function initialProjectTextSearchState(state) {
  return {
    query: state?.query || "",
    results: [],
    ongoingSearch: null,
    status: statusType.initial,
  };
}

function update(state = initialProjectTextSearchState(), action) {
  switch (action.type) {
    case "ADD_QUERY":
      return { ...state, query: action.query };

    case "ADD_SEARCH_RESULT":
      const { location, matches } = action;
      if (matches.length === 0) {
        return state;
      }

      const result = {
        type: "RESULT",
        location,
        // `matches` are generated by project-search worker's `findSourceMatches` method
        matches: matches.map(m => ({
          type: "MATCH",
          location: createLocation({
            ...location,
            // `matches` only contain line and column
            // `location` will already refer to the right source/sourceActor
            line: m.line,
            column: m.column,
          }),
          matchIndex: m.matchIndex,
          match: m.match,
          value: m.value,
        })),
      };
      return { ...state, results: [...state.results, result] };

    case "UPDATE_STATUS":
      const ongoingSearch =
        action.status == statusType.fetching ? state.ongoingSearch : null;
      return { ...state, status: action.status, ongoingSearch };

    case "CLEAR_SEARCH_RESULTS":
      return { ...state, results: [] };

    case "ADD_ONGOING_SEARCH":
      return { ...state, ongoingSearch: action.ongoingSearch };

    case "CLEAR_SEARCH":
    case "CLOSE_PROJECT_SEARCH":
      return initialProjectTextSearchState();
    case "NAVIGATE":
      return initialProjectTextSearchState(state);
  }
  return state;
}

export default update;
